//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.Controls 1.0

import wesual.data.Core 1.0

PropertyEditor {
    id : editor

    property bool      multiline           : false
    property alias     maximumLineCount    : displayText.maximumLineCount
    property alias     font                : displayText.font
    property alias     horizontalAlignment : displayText.horizontalAlignment
    property alias     color               : displayText.color
    property alias     wrapMode            : displayText.wrapMode
    property string    text
    property int       textMargin          : 3
    property Component background          : defaultBackground
    property Component cursorDelegate
    property string    placeholderText     : qsTrId("357e62e8cc4983da")
    property alias     lineHeight          : displayText.lineHeight
    property alias     lineHeightMode      : displayText.lineHeightMode
    property bool      showBorder          : false
    property var       validator
    property bool      preventStealing     : true
    property alias     renderType          : displayText.renderType

    // needed for focus on tab functionality in dashboard
    property bool      enableExternTabHandling : false
    signal backtabPressed();
    signal tabPressed();

    readonly property alias containsMouse : mouseArea.containsMouse

    function commit() {
        if (!mouseArea.currentEditor)
            return;

        var text = mouseArea.currentEditor.text;
        if (!multiline) {
            text = text.replace(/\r|\n/, " ");
        }

        editor.editingFinished(text);

        if (mouseArea.currentEditor)
            mouseArea.currentEditor.destroy();
        mouseArea.currentEditor = null;
    }

    function createEditor() {
        if (mouseArea.currentEditor) {
            focusEditor();

            return;
        }

        var realSingleLine =
                (editor.wrapMode === Text.NoWrap) && !editor.multiline;
        var edit = null;

        if (realSingleLine) {
            edit = singleLineEdit.createObject(mouseArea);
            edit.text = text;
        } else {
            edit = multiLineEdit.createObject(mouseArea);
            edit.text = text;
            editor.setLineHeight(edit.textDocument, editor.lineHeight);
        }

        mouseArea.currentEditor = edit;

        focusEditor();
    }

    function focusEditor() {
        mouseArea.currentEditor.forceActiveFocus();
        mouseArea.currentEditor.cursorPosition = editor.text.length;
        mouseArea.currentEditor.selectAll();
    }

    signal editingFinished(string text)

    FontMetrics {
        id : metrics
        font : editor.font
    }

    implicitWidth : Math.max(16, displayText.implicitWidth) + 2 * textMargin
    implicitHeight : {
        var h = displayText.contentHeight;

        if (mouseArea.currentEditor) {
            h = mouseArea.currentEditor.contentHeight;
        }

        if (editor.lineHeight < 1.0) {
            h += metrics.height - (metrics.height * editor.lineHeight);
        }

        return h + 2 * textMargin;
    }
    baselineOffset : displayText.y + displayText.baselineOffset
    //clip : true

    onBeginEdit : createEditor()
    onEndEdit   : commit()

    Component.onCompleted : {
        if (background) {
            var bg = background.createObject(editor);

            if (bg) {
                bg.z = -1;
                bg.anchors.fill = editor;
            }
        }
    }

    Keys.onUpPressed : {}
    Keys.onDownPressed : {}
    Keys.onRightPressed : {}
    Keys.onLeftPressed : {}
    Keys.onPressed : {
        if (event.key === Qt.Key_PageUp || event.key === Qt.Key_PageDown) {
            // do not propagate navigation keys while focus item
            event.accepted = true;
        }
    }

    MouseArea {
        id : mouseArea

        property Item currentEditor

        z : 2

        hoverEnabled : true
        anchors.fill : displayText
        onPressed : {
            editor.forceActiveFocus();
            mouseArea.preventStealing = editor.preventStealing;
        }
        cursorShape : Qt.IBeamCursor
    }

    Component {
        id : singleLineEdit

        TextInput {
            anchors.fill : parent
            selectByMouse : true
            z : 2
            font : editor.font
            color : editor.color
            selectionColor: UI.color(UI.TextSelection)
            selectedTextColor : UI.color(UI.SelectedTextColor)
            activeFocusOnTab : false
            cursorDelegate : editor.cursorDelegate
            horizontalAlignment : editor.horizontalAlignment
            wrapMode : Text.NoWrap
            validator : editor.validator || null
            clip : true
            renderType : displayText.renderType
            Keys.onReturnPressed : {
                commit();
                editor.focus = false;
            }
            Keys.onEnterPressed : {
                commit();
                editor.focus = false;
            }
            Keys.onEscapePressed : {
                undo();
                editor.focus = false;
            }
            Keys.onBacktabPressed : {
                if (enableExternTabHandling) {
                    editor.backtabPressed();
                    return;
                }

                var next = nextItemInFocusChain(false);
                if (next)
                    next.focus = true;
            }
            Keys.onTabPressed : {
                if (enableExternTabHandling) {
                    editor.tabPressed();
                    return;
                }

                var next = nextItemInFocusChain(true);
                if (next)
                    next.focus = true;
            }
        }
    }

    Component {
        id : multiLineEdit

        TextEdit {
            width : parent.width
            selectByMouse : true
            z : 2
            font : editor.font
            color : editor.color
            textFormat : Text.PlainText
            selectionColor: UI.color(UI.TextSelection)
            selectedTextColor : UI.color(UI.SelectedTextColor)
            activeFocusOnTab : false
            cursorDelegate : editor.cursorDelegate
            horizontalAlignment : editor.horizontalAlignment
            wrapMode : TextEdit.WordWrap
            renderType : displayText.renderType
            Keys.onReturnPressed : {
                if (editor.multiline) {
                    event.accepted = false;
                } else {
                    commit();
                    editor.focus = false;
                }
            }
            Keys.onEnterPressed : {
                if (editor.multiline) {
                    event.accepted = false;
                } else {
                    commit();
                    editor.focus = false;
                }
            }
            Keys.onBacktabPressed : {
                if (enableExternTabHandling) {
                    editor.backtabPressed();
                    return;
                }

                var next = nextItemInFocusChain(false);
                if (next)
                    next.focus = true;
            }
            Keys.onTabPressed : {
                if (enableExternTabHandling) {
                    editor.tabPressed();
                    return;
                }

                var next = nextItemInFocusChain(true);
                if (next)
                    next.focus = true;
            }
            Keys.onEscapePressed : {
                // undo
                text = editor.text;
                editor.focus = false;
            }
        }
    }

    Component {
        id : defaultBackground

        Rectangle {
            id : background

            readonly property bool borderVisible :
                editor.showBorder || editor.containsMouse || editor.activeFocus

            anchors.fill : parent
            opacity : borderVisible ? 1 : 0
            border {
                width : 1
                color : UI.color(UI.PrimaryControlBorder)
            }

            Behavior on opacity {
                NumberAnimation { duration : 0 }
            }

            states : [
                State {
                    name : "selected"
                    when : editor.activeFocus

                    PropertyChanges {
                        target : background.border
                        color  : UI.color(UI.PrimaryPress)
                    }
                },
                State {
                    name : "hovered"
                    when : editor.containsMouse

                    PropertyChanges {
                        target : background.border
                        color  : UI.color(UI.PrimaryHover)
                    }
                }
            ]
        }
    }

    Text {
        id : displayText

        text : parent.text.length > 0 ? parent.text : parent.placeholderText
        visible : mouseArea.currentEditor === null
        y : editor.textMargin
        anchors {
            left : parent.left
            right : parent.right
            margins: editor.textMargin
        }
        elide : Text.ElideRight
        font.family : UI.fontFamily(UI.PrimaryFont)
        font.weight : UI.fontWeight(UI.PrimaryFont)
        font.pixelSize : 12
        textFormat : Text.PlainText

        states : State {
            name : "placeholder"
            when : editor.text.length === 0 && !mouseArea.currentEditor

            PropertyChanges {
                target : displayText
                color  : UI.color(UI.SecondaryTextColor)
            }
        }
    }
}
